<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Check if user has permission to view suppliers
// Check if user has any supplier-related permissions
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions)) {
    $_SESSION['error'] = "You don't have permission to access supplier management.";
    header('Location: dashboard.php');
    exit;
}

if (false) { // Disable the old check
    // Permission modal will be shown instead of redirecting
}

$activePage = 'suppliers';
$success = '';
$error = '';

// Handle supplier deletion
if (isset($_POST['delete_supplier']) && (hasPermission('manage_suppliers') || $isSuperAdmin)) {
    try {
        $stmt = $pdo->prepare("DELETE FROM suppliers WHERE id = ?");
        $stmt->execute([$_POST['supplier_id']]);
        $success = "Supplier deleted successfully!";
    } catch (PDOException $e) {
        $error = "Error deleting supplier: " . $e->getMessage();
    }
}

// Handle supplier addition/update
if (isset($_POST['save_supplier']) && (hasPermission('manage_suppliers') || $isSuperAdmin)) {
    try {
        $name = trim($_POST['name']);
        $contact_person = trim($_POST['contact_person']);
        $phone = trim($_POST['phone']);
        $email = trim($_POST['email']);
        $address = trim($_POST['address']);
        $tax_number = trim($_POST['tax_number']);

        if (empty($name)) {
            throw new Exception("Supplier name is required");
        }

        if (isset($_POST['supplier_id']) && !empty($_POST['supplier_id'])) {
            // Update existing supplier
            $stmt = $pdo->prepare("
                UPDATE suppliers 
                SET name = ?, 
                    contact_person = ?, 
                    phone = ?, 
                    email = ?, 
                    address = ?, 
                    tax_number = ? 
                WHERE id = ?
            ");
            $stmt->execute([$name, $contact_person, $phone, $email, $address, $tax_number, $_POST['supplier_id']]);
            $success = "Supplier updated successfully!";
        } else {
            // Add new supplier
            $stmt = $pdo->prepare("
                INSERT INTO suppliers (
                    name, 
                    contact_person, 
                    phone, 
                    email, 
                    address, 
                    tax_number
                ) VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$name, $contact_person, $phone, $email, $address, $tax_number]);
            $success = "Supplier added successfully!";
        }

        // Log the action
        logSystemAction(
            isset($_POST['supplier_id']) ? 'supplier_updated' : 'supplier_created',
            isset($_POST['supplier_id']) ? "Updated supplier: {$name}" : "Added new supplier: {$name}"
        );

        // Redirect to refresh the page
        header("Location: suppliers.php?success=" . urlencode($success));
        exit;
    } catch (Exception $e) {
        $error = "Error saving supplier: " . $e->getMessage();
    }
}

// Get all suppliers with their items and payment details
$stmt = $pdo->query("
    SELECT 
        s.*,
        COUNT(sr.id) as total_items,
        SUM(CASE WHEN sr.payment_status = 'paid' THEN 1 ELSE 0 END) as paid_items,
        SUM(sr.quantity * sr.unit_price) as total_amount,
        SUM(CASE WHEN sr.payment_status = 'paid' THEN sr.quantity * sr.unit_price ELSE 0 END) as paid_amount,
        MAX(CASE WHEN sr.payment_status = 'paid' THEN sr.updated_at END) as last_payment_date,
        MIN(CASE WHEN sr.payment_status = 'pending' THEN sr.created_at END) as oldest_pending_date
    FROM suppliers s
    LEFT JOIN supplier_receipts sr ON s.id = sr.supplier_id
    GROUP BY s.id
    ORDER BY s.name
");
$suppliers = $stmt->fetchAll();

// Get all products for the Create New Receipt modal
$stmt = $pdo->query("SELECT id, name FROM products WHERE status = 'active' ORDER BY name");
$products = $stmt->fetchAll();

// Get all items for each supplier (receipts and manual adjustments)
$supplier_items = [];
foreach ($suppliers as $supplier) {
    // Supplier Receipts
    $stmt = $pdo->prepare("
        SELECT 
            sr.id,
            sr.created_at,
            p.name as product_name,
            sr.batch_number,
            sr.quantity,
            sr.unit_price,
            (sr.quantity * sr.unit_price) as total,
            sr.payment_status,
            u.username as entered_by,
            'Receipt' as source,
            sr.serial_numbers,
            sr.payment_status as payment_status_raw,
            sr.payment_status as payment_status_display
        FROM supplier_receipts sr
        JOIN products p ON sr.product_id = p.id
        JOIN users u ON sr.created_by = u.id
        WHERE sr.supplier_id = ?
        ORDER BY sr.created_at DESC
    ");
    $stmt->execute([$supplier['id']]);
    $receipts = $stmt->fetchAll();

    // Manual Stock Adjustments
    $stmt = $pdo->prepare("
        SELECT 
            sa.id,
            sa.adjusted_at as created_at,
            p.name as product_name,
            NULL as batch_number,
            sa.quantity,
            NULL as unit_price,
            NULL as total,
            NULL as payment_status,
            u.username as entered_by,
            'Manual Adjustment' as source,
            NULL as serial_numbers,
            NULL as payment_status_raw,
            NULL as payment_status_display
        FROM stock_adjustments sa
        JOIN products p ON sa.product_id = p.id
        JOIN users u ON sa.adjusted_by = u.id
        WHERE sa.supplier_id = ?
        ORDER BY sa.adjusted_at DESC
    ");
    $stmt->execute([$supplier['id']]);
    $adjustments = $stmt->fetchAll();

    // Merge and sort by created_at DESC
    $all_items = array_merge($receipts, $adjustments);
    usort($all_items, function($a, $b) {
        return strtotime($b['created_at']) <=> strtotime($a['created_at']);
    });
    $supplier_items[$supplier['id']] = $all_items;
}

// Get store information
$stmt = $pdo->prepare("
    SELECT setting_key, setting_value 
    FROM store_settings 
    WHERE setting_group = 'store_info' 
    AND setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'currency', 'store_logo', 'website')
");
$stmt->execute();
$storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Check for success message in URL
if (isset($_GET['success'])) {
    $success = $_GET['success'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Suppliers - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css" />
    <style>
        :root {
            --primary-color: #3b82f6;
            --primary-dark: #2563eb;
            --secondary-color: #64748b;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #06b6d4;
            --light-bg: #f8fafc;
            --border-color: #e2e8f0;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        body { 
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); 
            font-family: 'Inter', sans-serif;
            color: var(--text-primary);
            line-height: 1.6;
        }

        /* Modern Card Design */
        .card { 
            border-radius: 16px; 
            box-shadow: var(--shadow-md);
            border: none;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            transition: all 0.3s ease;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .card-header { 
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border-radius: 16px 16px 0 0 !important;
            border-bottom: none;
            padding: 1.5rem;
        }

        .card-header h4 {
            margin: 0;
            font-weight: 700;
            font-size: 1.5rem;
        }

        .card-body {
            padding: 2rem;
        }

        /* Modern Button Design */
        .btn {
            border-radius: 12px;
            font-weight: 600;
            padding: 0.75rem 1.5rem;
            transition: all 0.3s ease;
            border: none;
            position: relative;
            overflow: hidden;
        }

        .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }

        .btn:hover::before {
            left: 100%;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, var(--primary-dark), #1d4ed8);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .btn-success {
            background: linear-gradient(135deg, var(--success-color), #059669);
            color: white;
        }

        .btn-info {
            background: linear-gradient(135deg, var(--info-color), #0891b2);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, var(--warning-color), #d97706);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, var(--danger-color), #dc2626);
            color: white;
        }

        .btn-secondary {
            background: linear-gradient(135deg, var(--secondary-color), #475569);
            color: white;
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 8px;
        }

        /* Modern Table Design */
        .table {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: var(--shadow-sm);
            margin-bottom: 0;
        }

        .table th {
            background: linear-gradient(135deg, var(--light-bg), #f1f5f9);
            border-bottom: 2px solid var(--border-color);
            color: var(--text-primary);
            font-weight: 700;
            padding: 1rem;
            text-transform: uppercase;
            font-size: 0.875rem;
            letter-spacing: 0.05em;
        }

        .table td {
            padding: 1rem;
            vertical-align: middle;
            color: var(--text-secondary);
            border-bottom: 1px solid var(--border-color);
        }

        .table tbody tr {
            transition: all 0.2s ease;
        }

        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8fafc, #f1f5f9);
            transform: scale(1.01);
        }

        /* Supplier Header Row */
        .supplier-header {
            background: linear-gradient(135deg, #ffffff, #f8fafc);
            border-left: 4px solid var(--primary-color);
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .supplier-header:hover {
            background: linear-gradient(135deg, #f8fafc, #e0e7ff);
            transform: translateX(5px);
        }

        .supplier-toggle {
            transition: transform 0.3s ease;
            color: var(--primary-color);
        }

        .supplier-header.expanded .supplier-toggle {
            transform: rotate(90deg);
        }

        /* Modern Form Controls */
        .form-control, .form-select {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.875rem 1rem;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            transform: translateY(-1px);
        }

        .input-group-text {
            border-radius: 12px 0 0 12px;
            border: 2px solid var(--border-color);
            background: var(--light-bg);
            color: var(--text-secondary);
        }

        /* Modern Modal Design */
        .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: var(--shadow-lg);
            overflow: hidden;
        }

        .modal-header {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border-bottom: none;
            padding: 1.5rem;
        }

        .modal-title {
            font-weight: 700;
            font-size: 1.25rem;
        }

        .modal-body {
            padding: 2rem;
        }

        .modal-footer {
            background: var(--light-bg);
            border-top: 1px solid var(--border-color);
            padding: 1.5rem;
        }

        /* Badge Design */
        .badge {
            border-radius: 8px;
            font-weight: 600;
            padding: 0.5rem 0.75rem;
            font-size: 0.75rem;
        }

        .bg-success {
            background: linear-gradient(135deg, var(--success-color), #059669) !important;
        }

        .bg-warning {
            background: linear-gradient(135deg, var(--warning-color), #d97706) !important;
        }

        .bg-secondary {
            background: linear-gradient(135deg, var(--secondary-color), #475569) !important;
        }

        .bg-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark)) !important;
        }

        .bg-info {
            background: linear-gradient(135deg, var(--info-color), #0891b2) !important;
        }

        /* Alert Design */
        .alert {
            border-radius: 12px;
            border: none;
            padding: 1rem 1.5rem;
            font-weight: 500;
        }

        .alert-success {
            background: linear-gradient(135deg, #d1fae5, #a7f3d0);
            color: #065f46;
        }

        .alert-danger {
            background: linear-gradient(135deg, #fee2e2, #fecaca);
            color: #991b1b;
        }

        /* Search Box */
        .search-container {
            position: relative;
        }

        .search-container .input-group {
            border-radius: 12px;
            overflow: hidden;
            box-shadow: var(--shadow-sm);
        }

        .search-container .form-control {
            border: none;
            padding-left: 0;
        }

        .search-container .input-group-text {
            border: none;
            background: white;
        }

        /* Payment Distribution Table */
        .payment-distribution-container {
            max-height: 400px;
            overflow-y: auto;
            border-radius: 12px;
            border: 2px solid var(--border-color);
            background: white;
        }

        .payment-distribution-container table {
            margin-bottom: 0;
        }

        .payment-distribution-container thead th {
            position: sticky;
            top: 0;
            background: linear-gradient(135deg, var(--light-bg), #f1f5f9);
            z-index: 10;
            border-bottom: 2px solid var(--border-color);
        }

        /* Statistics Cards */
        .stats-card {
            background: linear-gradient(135deg, white, #f8fafc);
            border-radius: 16px;
            padding: 1.5rem;
            box-shadow: var(--shadow-md);
            border-left: 4px solid var(--primary-color);
            transition: all 0.3s ease;
        }

        .stats-card:hover {
            transform: translateY(-3px);
            box-shadow: var(--shadow-lg);
        }

        .stats-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary-color);
            margin-bottom: 0.5rem;
        }

        .stats-label {
            color: var(--text-secondary);
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.875rem;
            letter-spacing: 0.05em;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .card-body {
                padding: 1rem;
            }
            
            .btn {
                padding: 0.5rem 1rem;
                font-size: 0.875rem;
            }
            
            .table-responsive {
                border-radius: 12px;
                overflow: hidden;
            }
        }

        /* Animation Classes */
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .slide-in {
            animation: slideIn 0.3s ease-out;
        }

        @keyframes slideIn {
            from { transform: translateX(-100%); }
            to { transform: translateX(0); }
        }

        /* Custom Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
        }

        ::-webkit-scrollbar-track {
            background: var(--light-bg);
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: linear-gradient(135deg, var(--primary-dark), #1d4ed8);
        }

        /* Ripple Effect */
        .btn {
            position: relative;
            overflow: hidden;
        }

        .ripple {
            position: absolute;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.6);
            transform: scale(0);
            animation: ripple-animation 0.6s linear;
            pointer-events: none;
        }

        @keyframes ripple-animation {
            to {
                transform: scale(4);
                opacity: 0;
            }
        }

        /* Enhanced Form Validation */
        .form-control.is-valid {
            border-color: var(--success-color);
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 8 8'%3e%3cpath fill='%2310b981' d='m2.3 6.73.94-.94 2.89 2.89 2.89-2.89.94.94L6.12 9.62z'/%3e%3c/svg%3e");
            background-repeat: no-repeat;
            background-position: right calc(0.375em + 0.1875rem) center;
            background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
        }

        .form-control.is-invalid {
            border-color: var(--danger-color);
            background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 12 12' width='12' height='12' fill='none' stroke='%23ef4444'%3e%3ccircle cx='6' cy='6' r='4.5'/%3e%3cpath d='m5.8 4.6 1.4 1.4m0-1.4L5.8 6'/%3e%3c/svg%3e");
            background-repeat: no-repeat;
            background-position: right calc(0.375em + 0.1875rem) center;
            background-size: calc(0.75em + 0.375rem) calc(0.75em + 0.375rem);
        }

        /* Enhanced Progress Bars */
        .progress {
            height: 8px;
            border-radius: 4px;
            background: var(--light-bg);
            overflow: hidden;
        }

        .progress-bar {
            border-radius: 4px;
            background: linear-gradient(90deg, var(--success-color), #059669);
            transition: width 0.6s ease;
        }

        /* Enhanced Badges */
        .badge {
            position: relative;
            overflow: hidden;
        }

        .badge::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            transition: left 0.5s;
        }

        .badge:hover::before {
            left: 100%;
        }

        /* Enhanced Table Hover Effects */
        .table tbody tr {
            position: relative;
        }

        .table tbody tr::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.05), rgba(37, 99, 235, 0.05));
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
        }

        .table tbody tr:hover::after {
            opacity: 1;
        }

        /* Enhanced Modal Animations */
        .modal.fade .modal-dialog {
            transition: transform 0.3s ease-out;
            transform: translate(0, -50px);
        }

        .modal.show .modal-dialog {
            transform: none;
        }

        /* Enhanced Button Group */
        .btn-group .btn {
            position: relative;
            z-index: 1;
        }

        .btn-group .btn:hover {
            z-index: 2;
        }

        /* Enhanced Search Container */
        .search-container .input-group:focus-within {
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            transform: translateY(-1px);
        }

        /* Enhanced Statistics Cards */
        .stats-card {
            position: relative;
            overflow: hidden;
        }

        .stats-card::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(59, 130, 246, 0.1), transparent);
            transform: rotate(45deg);
            transition: transform 0.6s ease;
            opacity: 0;
        }

        .stats-card:hover::before {
            transform: rotate(45deg) translate(50%, 50%);
            opacity: 1;
        }

        /* Enhanced Alert Animations */
        .alert {
            animation: slideInRight 0.5s ease-out;
        }

        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        /* Enhanced Loading States */
        .btn:disabled {
            opacity: 0.7;
            cursor: not-allowed;
        }

        .btn:disabled .fa-spinner {
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        /* Enhanced Tooltips */
        .tooltip {
            font-size: 0.875rem;
        }

        .tooltip-inner {
            background: var(--text-primary);
            border-radius: 8px;
            padding: 0.5rem 0.75rem;
        }

        /* Enhanced Focus States */
        .btn:focus,
        .form-control:focus,
        .form-select:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.25);
        }

        /* Enhanced Disabled States */
        .btn:disabled,
        .form-control:disabled,
        .form-select:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        /* Enhanced Print Styles */
        @media print {
            .btn,
            .search-container,
            .modal {
                display: none !important;
            }
            
            .card {
                box-shadow: none !important;
                border: 1px solid #000 !important;
            }
            
            .table {
                border: 1px solid #000 !important;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <div class="card mb-4 fade-in">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="fas fa-truck me-2"></i>Suppliers Management
                        </h4>
                        <div class="d-flex align-items-center gap-3">
                            <div class="search-container">
                                <div class="input-group" style="width: 300px;">
                                    <span class="input-group-text">
                                        <i class="fas fa-search"></i>
                                    </span>
                                <input type="text" id="supplierSearch" class="form-control" placeholder="Search suppliers...">
                                </div>
                            </div>
                            <?php if (hasPermission('manage_suppliers')): ?>
                            <div class="d-flex gap-2">
                                <a href="viewsuppliers.php" class="btn btn-info">
                                    <i class="fas fa-list me-2"></i>View Suppliers
                                </a>
                                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#supplierModal">
                                    <i class="fas fa-plus me-2"></i>Add Supplier
                                </button>
                                <a href="receivesupplieritem.php" class="btn btn-info">
                                    <i class="fas fa-receipt me-2"></i>Supplier Receipts
                                </a>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="card-body pb-0">
                        <!-- Statistics Cards -->
                        <div class="row mb-4 fade-in">
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <div class="stats-value"><?php echo count($suppliers); ?></div>
                                    <div class="stats-label">Total Suppliers</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <div class="stats-value"><?php echo number_format(array_sum(array_column($suppliers, 'total_amount')), 2); ?></div>
                                    <div class="stats-label">Total Amount</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <div class="stats-value"><?php echo number_format(array_sum(array_column($suppliers, 'paid_amount')), 2); ?></div>
                                    <div class="stats-label">Total Paid</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <div class="stats-value"><?php echo number_format(array_sum(array_column($suppliers, 'total_amount')) - array_sum(array_column($suppliers, 'paid_amount')), 2); ?></div>
                                    <div class="stats-label">Outstanding</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card mb-4 fade-in">
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success fade-in">
                                <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                            </div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger fade-in">
                                <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                            </div>
                        <?php endif; ?>

                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th><i class="fas fa-building me-2"></i>Supplier</th>
                                        <th><i class="fas fa-user me-2"></i>Contact</th>
                                        <th><i class="fas fa-boxes me-2"></i>Total Items</th>
                                        <th><i class="fas fa-money-bill-wave me-2"></i>Total Amount</th>
                                        <th><i class="fas fa-check-circle me-2"></i>Total Paid</th>
                                        <th><i class="fas fa-exclamation-triangle me-2"></i>Total Balance</th>
                                        <th><i class="fas fa-calendar me-2"></i>Payment Dates</th>
                                        <th><i class="fas fa-chart-pie me-2"></i>Payment Status</th>
                                        <th><i class="fas fa-cogs me-2"></i>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <tr class="supplier-header" data-supplier-id="<?php echo $supplier['id']; ?>" style="cursor: pointer;">
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <i class="fas fa-chevron-right me-3 supplier-toggle"></i>
                                                    <div>
                                                        <strong><?php echo htmlspecialchars($supplier['name']); ?></strong>
                                                        <?php if ($supplier['tax_number']): ?>
                                                            <br><small class="text-muted">Tax: <?php echo htmlspecialchars($supplier['tax_number']); ?></small>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($supplier['contact_person']); ?></strong>
                                                    <br><small class="text-muted">
                                                        <i class="fas fa-phone me-1"></i><?php echo htmlspecialchars($supplier['phone']); ?>
                                                        <?php if ($supplier['email']): ?>
                                                            <br><i class="fas fa-envelope me-1"></i><?php echo htmlspecialchars($supplier['email']); ?>
                                                        <?php endif; ?>
                                                    </small>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge bg-primary"><?php echo $supplier['total_items']; ?></span>
                                            </td>
                                            <td>
                                                <strong class="text-primary"><?php echo number_format($supplier['total_amount'], 2); ?></strong>
                                            </td>
                                            <td>
                                                <strong class="text-success"><?php echo number_format($supplier['paid_amount'], 2); ?></strong>
                                            </td>
                                            <td>
                                                <?php 
                                                $balance = $supplier['total_amount'] - $supplier['paid_amount'];
                                                $balanceClass = $balance > 0 ? 'text-danger' : 'text-success';
                                                ?>
                                                <strong class="<?php echo $balanceClass; ?>"><?php echo number_format($balance, 2); ?></strong>
                                            </td>
                                            <td>
                                                <?php if ($supplier['last_payment_date']): ?>
                                                    <div class="text-success">
                                                        <i class="fas fa-check me-1"></i>Last: <?php echo date('M d, Y', strtotime($supplier['last_payment_date'])); ?>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($supplier['oldest_pending_date']): ?>
                                                    <div class="text-warning">
                                                        <i class="fas fa-clock me-1"></i>Pending: <?php echo date('M d, Y', strtotime($supplier['oldest_pending_date'])); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php
                                                $paid_count = $supplier['paid_items'];
                                                $total_count = $supplier['total_items'];
                                                $paid_amount = $supplier['paid_amount'];
                                                $total_amount = $supplier['total_amount'];
                                                $percentage = $total_amount > 0 ? ($paid_amount / $total_amount) * 100 : 0;
                                                ?>
                                                <div class="d-flex align-items-center">
                                                    <div class="progress me-2" style="width: 60px; height: 8px;">
                                                        <div class="progress-bar bg-success" style="width: <?php echo $percentage; ?>%"></div>
                                                    </div>
                                                    <small><?php echo round($percentage, 1); ?>%</small>
                                                </div>
                                                <small class="text-muted"><?php echo $paid_count; ?>/<?php echo $total_count; ?> items</small>
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <button type="button" class="btn btn-success btn-sm pay-btn" 
                                                        data-supplier-id="<?php echo $supplier['id']; ?>"
                                                        data-supplier-name="<?php echo htmlspecialchars($supplier['name']); ?>"
                                                        data-total-amount="<?php echo $supplier['total_amount']; ?>"
                                                        data-paid-amount="<?php echo $supplier['paid_amount']; ?>"
                                                        data-items='<?php echo json_encode(array_map(function($item) {
                                                            return [
                                                                'id' => $item['id'],
                                                                'created_at' => $item['created_at'],
                                                                'product_name' => $item['product_name'],
                                                                'quantity' => $item['quantity'],
                                                                'unit_price' => $item['unit_price'],
                                                                'payment_status' => $item['payment_status']
                                                            ];
                                                        }, $supplier_items[$supplier['id']])); ?>'
                                                        data-permission="process_supplier_payment" data-permission-label="Process Supplier Payment">
                                                        <i class="fas fa-money-bill"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-info btn-sm" onclick="event.stopPropagation(); editSupplier(<?php echo htmlspecialchars(json_encode($supplier)); ?>)" data-permission="edit_supplier" data-permission-label="Edit Supplier">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-danger btn-sm" onclick="event.stopPropagation(); confirmDeleteSupplier(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>', <?php echo $supplier['total_amount']; ?>, <?php echo $supplier['paid_amount']; ?>)" data-permission="delete_supplier" data-permission-label="Delete Supplier">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-secondary btn-sm" onclick="event.stopPropagation(); openPrintModal(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>');" data-permission="view_suppliers" data-permission-label="View Suppliers">
                                                        <i class="fas fa-print"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <tr class="supplier-details" data-supplier-id="<?php echo $supplier['id']; ?>" style="display: none;">
                                            <td colspan="9">
                                                <div class="card border-0 shadow-sm">
                                                    <div class="card-body">
                                                        <h6 class="card-title mb-3">
                                                            <i class="fas fa-list me-2"></i>Transaction History
                                                        </h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm table-bordered">
                                                                <thead class="table-light">
                                                            <tr>
                                                                <th>Date</th>
                                                                <th>Product</th>
                                                                <th>Batch Number</th>
                                                                <th>Qty</th>
                                                                <th>Price</th>
                                                                <th>Total</th>
                                                                <th>Source</th>
                                                                <th>Payment Status</th>
                                                                <th>Entered By</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($supplier_items[$supplier['id']] as $item): ?>
                                                                <tr>
                                                                    <td><?php echo date('Y-m-d H:i', strtotime($item['created_at'])); ?></td>
                                                                    <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                                                                    <td><?php echo htmlspecialchars($item['batch_number'] ?? ''); ?></td>
                                                                    <td><?php echo number_format($item['quantity'], 2); ?></td>
                                                                    <td><?php echo $item['unit_price'] !== null ? number_format($item['unit_price'], 2) : '-'; ?></td>
                                                                    <td><?php echo $item['total'] !== null ? number_format($item['total'], 2) : '-'; ?></td>
                                                                    <td><span class="badge bg-<?php echo $item['source'] === 'Receipt' ? 'primary' : 'warning'; ?>"><?php echo $item['source']; ?></span></td>
                                                                    <td>
                                                                        <?php if ($item['source'] === 'Receipt'): ?>
                                                                            <span class="badge bg-<?php 
                                                                                echo $item['payment_status_raw'] === 'paid' ? 'success' : 
                                                                                    ($item['payment_status_raw'] === 'credit' ? 'warning' : 'secondary'); 
                                                                            ?>">
                                                                                <?php echo $item['payment_status_display']; ?>
                                                                            </span>
                                                                        <?php else: ?>
                                                                            <span class="badge bg-info">N/A</span>
                                                                        <?php endif; ?>
                                                                    </td>
                                                                    <td><?php echo htmlspecialchars($item['entered_by']); ?></td>
                                                                </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Supplier Modal -->
    <div class="modal fade" id="supplierModal" tabindex="-1" aria-labelledby="supplierModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="supplierModalLabel">Add Supplier</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" class="needs-validation" novalidate>
                    <div class="modal-body">
                        <input type="hidden" name="supplier_id" id="supplier_id">
                        <div class="mb-3">
                            <label for="name" class="form-label">Supplier Name *</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                            <div class="invalid-feedback">Please enter supplier name.</div>
                        </div>
                        <div class="mb-3">
                            <label for="contact_person" class="form-label">Contact Person</label>
                            <input type="text" class="form-control" id="contact_person" name="contact_person">
                        </div>
                        <div class="mb-3">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" class="form-control" id="phone" name="phone">
                        </div>
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email">
                        </div>
                        <div class="mb-3">
                            <label for="address" class="form-label">Address</label>
                            <textarea class="form-control" id="address" name="address" rows="2"></textarea>
                        </div>
                        <div class="mb-3">
                            <label for="tax_number" class="form-label">Tax Number</label>
                            <input type="text" class="form-control" id="tax_number" name="tax_number">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="save_supplier" class="btn btn-primary">Save Supplier</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Payment Modal -->
    <div class="modal fade" id="paymentModal" tabindex="-1" aria-labelledby="paymentModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="paymentModalLabel">
                        <i class="fas fa-money-bill-wave me-2"></i>Payment Details
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="paymentForm" method="post">
                        <input type="hidden" name="supplier_id" id="payment_supplier_id">
                        
                        <!-- Balance and Payment Method Section -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="card h-100 border-0 shadow-sm">
                                    <div class="card-body">
                                        <label class="form-label text-muted small">Total Balance</label>
                                        <h3 class="mb-0 text-primary" id="total_balance">0.00</h3>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card h-100 border-0 shadow-sm">
                                    <div class="card-body">
                                        <label class="form-label text-muted small">Payment Method</label>
                                        <select class="form-select form-select-lg" name="payment_method" id="payment_method" required onchange="handlePaymentMethodChange()">
                                            <option value="">Select Payment Method</option>
                                            <option value="cash"><i class="fas fa-money-bill"></i> Cash</option>
                                            <option value="mpesa"><i class="fas fa-mobile-alt"></i> M-PESA</option>
                                            <option value="cheque"><i class="fas fa-money-check"></i> Cheque</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Method Details Section -->
                        <div id="mpesaSection" class="card mb-4 border-0 shadow-sm" style="display: none;">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <i class="fas fa-mobile-alt text-primary me-2"></i>
                                    <h6 class="mb-0">M-PESA Details</h6>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Transaction Code</label>
                                    <input type="text" class="form-control form-control-lg" name="mpesa_code" id="mpesa_code" placeholder="Enter M-PESA transaction code">
                                </div>
                            </div>
                        </div>

                        <div id="chequeSection" class="card mb-4 border-0 shadow-sm" style="display: none;">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <i class="fas fa-money-check text-primary me-2"></i>
                                    <h6 class="mb-0">Cheque Details</h6>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Cheque Number</label>
                                    <input type="text" class="form-control form-control-lg" name="cheque_number" id="cheque_number" placeholder="Enter cheque number">
                                </div>
                            </div>
                        </div>

                        <!-- Amount and Notes Section -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="card border-0 shadow-sm">
                                    <div class="card-body">
                                        <label class="form-label">Payment Amount</label>
                                        <div class="input-group input-group-lg">
                                            <span class="input-group-text">KSH</span>
                                            <input type="number" class="form-control" name="amount" id="payment_amount" step="any" min="0.01" required onchange="updatePaymentDistribution()">
                                        </div>
                                        <div class="invalid-feedback">Amount cannot exceed total balance.</div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Distribution Section -->
                        <div class="card border-0 shadow-sm">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <i class="fas fa-chart-pie text-primary me-2"></i>
                                    <h6 class="mb-0">Payment Distribution</h6>
                                </div>
                                <div class="payment-distribution-container">
                                    <div class="table-responsive">
                                        <table class="table table-hover" id="paymentDistributionTable">
                                            <thead class="table-light">
                                                <tr>
                                                    <th>Date</th>
                                                    <th>Product</th>
                                                    <th>Total Amount</th>
                                                    <th>Paid Amount</th>
                                                    <th>Remaining</th>
                                                    <th>This Payment</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-light" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Close
                    </button>
                    <button type="button" class="btn btn-primary" onclick="submitPayment()" data-permission="process_supplier_payment" data-permission-label="Process Supplier Payment">
                        <i class="fas fa-save me-2"></i>Save Payment
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteConfirmModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Delete Supplier</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="deleteWarning" class="alert alert-warning" style="display: none;">
                        <i class="fas fa-exclamation-triangle"></i>
                        <strong>Warning:</strong> You can't delete supplier, You have Pending Payments To Supplier!!<br>
                        <span id="amountOwed" class="mt-2 d-block"></span>
                    </div>
                    <div id="deleteConfirmation">
                        <p>Are you sure you want to delete supplier: <strong id="deleteSupplierName"></strong>?</p>
                        <p class="text-danger">This action cannot be undone.</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-danger" id="deleteButton" onclick="deleteSupplier()">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment History Modal -->
    <div class="modal fade" id="paymentHistoryModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Payment History for <span id="phSupplierName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form class="row g-3 mb-3" id="phFilterForm">
                        <div class="col-md-4">
                            <label class="form-label">Start Date</label>
                            <input type="date" name="start_date" class="form-control">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">End Date</label>
                            <input type="date" name="end_date" class="form-control">
                        </div>
                        <div class="col-md-4 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary w-100">Search</button>
                        </div>
                    </form>
                    <div class="table-responsive">
                        <table class="table table-bordered" id="phTable">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Payment Amount</th>
                                    <th>Payment Method</th>
                                    <th>Reference</th>
                                    <th>Balance After</th>
                                </tr>
                            </thead>
                            <tbody id="phTableBody">
                                <!-- Payment history rows will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                    <button class="btn btn-secondary mt-3" id="phPrintBtn"><i class="fas fa-print"></i> Print</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Print Options Modal -->
    <div class="modal fade" id="printOptionsModal" tabindex="-1" aria-labelledby="printOptionsModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="printOptionsModalLabel">Print Options for <span id="printSupplierName"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <button type="button" class="btn btn-primary w-100 mb-2" id="printReportBtn"><i class="fas fa-file-alt"></i> Print Supplier Report</button>
                    <button type="button" class="btn btn-info w-100" id="printPaymentHistoryBtn"><i class="fas fa-file-invoice-dollar"></i> Print Payment History</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Create New Receipt Modal -->
    <div class="modal fade" id="createReceiptModal" tabindex="-1" aria-labelledby="createReceiptModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="createReceiptModalLabel">Create New Receipt</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form method="post" class="needs-validation" novalidate id="createReceiptForm">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Supplier *</label>
                                <select class="form-select form-select-sm" name="supplier_id" required>
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Items *</label>
                                <table class="table table-bordered" id="itemsTable">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Batch Number</th>
                                            <th>Notes</th>
                                            <th>Serial Numbers</th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td>
                                                <select class="form-select form-select-sm" name="product_id[]" required>
                                                    <option value="">Select Product</option>
                                                    <?php foreach ($products as $product): ?>
                                                        <option value="<?php echo $product['id']; ?>"><?php echo htmlspecialchars($product['name']); ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </td>
                                            <td><input type="number" class="form-control form-control-sm" name="quantity[]" min="1" required></td>
                                            <td><input type="number" class="form-control form-control-sm" name="unit_price[]" step="0.01" min="0.01" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="batch_number[]" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="notes[]"></td>
                                            <td><input type="text" class="form-control form-control-sm" name="serial_numbers[]"></td>
                                            <td><button type="button" class="btn btn-danger btn-sm remove-item">&times;</button></td>
                                        </tr>
                                    </tbody>
                                </table>
                                <button type="button" class="btn btn-success btn-sm" id="addItemRow"><i class="fas fa-plus"></i> Add Another Item</button>
                            </div>
                        </div>
                        <div class="mt-4">
                            <button type="submit" name="receive_item" class="btn btn-primary" data-permission="add_supplier_item" data-permission-label="Add Supplier Item">
                                <i class="fas fa-save"></i> Create Receipt
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div class="modal fade" id="successModal" tabindex="-1" aria-labelledby="successModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-success text-white border-0">
                    <h5 class="modal-title d-flex align-items-center" id="successModalLabel">
                        <span class="me-2" style="font-size:2rem;">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" class="feather feather-check-circle animate__animated animate__bounceIn"><circle cx="12" cy="12" r="10"/><path d="M9 12l2 2l4-4"/></svg>
                        </span>
                        Success
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center py-4">
                    <p id="successModalMessage" class="mb-0 fs-5 text-success">Payment recorded successfully!</p>
                </div>
                <div class="modal-footer border-0 justify-content-center">
                    <button type="button" class="btn btn-success px-4" data-bs-dismiss="modal" onclick="window.location.reload();">OK</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Error Modal -->
    <div class="modal fade" id="errorModal" tabindex="-1" aria-labelledby="errorModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 shadow-lg">
                <div class="modal-header bg-danger text-white border-0">
                    <h5 class="modal-title d-flex align-items-center" id="errorModalLabel">
                        <span class="me-2" style="font-size:2rem;">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" class="feather feather-x-circle animate__animated animate__shakeX"><circle cx="12" cy="12" r="10"/><path d="M15 9l-6 6M9 9l6 6"/></svg>
                        </span>
                        Error
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center py-4">
                    <p id="errorModalMessage" class="mb-0 fs-5 text-danger">An error occurred.</p>
                </div>
                <div class="modal-footer border-0 justify-content-center">
                    <button type="button" class="btn btn-danger px-4" data-bs-dismiss="modal">OK</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Form validation
        (function () {
            'use strict'
            var forms = document.querySelectorAll('.needs-validation')
            Array.prototype.slice.call(forms).forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    if (!form.checkValidity()) {
                        event.preventDefault()
                        event.stopPropagation()
                    }
                    form.classList.add('was-validated')
                }, false)
            })
        })()

        // Add search functionality
        document.getElementById('supplierSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('.supplier-header');
            
            rows.forEach(row => {
                const supplierName = row.querySelector('td:first-child').textContent.toLowerCase();
                const contactPerson = row.querySelector('td:nth-child(2)').textContent.toLowerCase();
                
                if (supplierName.includes(searchTerm) || contactPerson.includes(searchTerm)) {
                    row.style.display = '';
                    row.classList.add('fade-in');
                } else {
                    row.style.display = 'none';
                    // Hide corresponding details row
                    const supplierId = row.dataset.supplierId;
                    const details = document.querySelector(`.supplier-details[data-supplier-id="${supplierId}"]`);
                    if (details) {
                        details.style.display = 'none';
                    }
                }
            });
        });

        // Handle edit supplier
        document.querySelectorAll('.edit-supplier').forEach(button => {
            button.addEventListener('click', function() {
                const supplier = JSON.parse(this.dataset.supplier);
                document.getElementById('supplier_id').value = supplier.id;
                document.getElementById('name').value = supplier.name;
                document.getElementById('contact_person').value = supplier.contact_person;
                document.getElementById('phone').value = supplier.phone;
                document.getElementById('email').value = supplier.email;
                document.getElementById('address').value = supplier.address;
                document.getElementById('tax_number').value = supplier.tax_number;
                document.getElementById('supplierModalLabel').textContent = 'Edit Supplier';
            });
        });

        // Reset form when modal is closed
        document.getElementById('supplierModal').addEventListener('hidden.bs.modal', function() {
            const form = this.querySelector('form');
            form.reset();
            form.classList.remove('was-validated');
            document.getElementById('supplier_id').value = '';
            document.getElementById('supplierModalLabel').textContent = 'Add Supplier';
        });

        // Add this to your existing JavaScript
        document.querySelectorAll('.supplier-header').forEach(header => {
            header.addEventListener('click', function(e) {
                // Don't expand if clicking on a button
                if (e.target.closest('.btn-group')) {
                    return;
                }
                
                const supplierId = this.dataset.supplierId;
                const details = document.querySelector(`.supplier-details[data-supplier-id="${supplierId}"]`);
                const toggle = this.querySelector('.supplier-toggle');
                
                // Add smooth animation
                if (details.style.display === 'none') {
                    details.style.display = 'table-row';
                    details.classList.add('slide-in');
                    this.classList.add('expanded');
                    toggle.style.transform = 'rotate(90deg)';
                } else {
                    details.classList.remove('slide-in');
                    this.classList.remove('expanded');
                    toggle.style.transform = 'rotate(0deg)';
                    setTimeout(() => {
                    details.style.display = 'none';
                    }, 300);
                }
            });
        });

        let unpaidItems = [];
        let paymentModal;

        document.addEventListener('DOMContentLoaded', function() {
            // Initialize the payment modal
            paymentModal = new bootstrap.Modal(document.getElementById('paymentModal'));

            // Add click handlers for pay buttons
            document.querySelectorAll('.pay-btn').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const supplierId = this.dataset.supplierId;
                    const supplierName = this.dataset.supplierName;
                    const totalAmount = parseFloat(this.dataset.totalAmount);
                    const paidAmount = parseFloat(this.dataset.paidAmount);
                    const items = JSON.parse(this.dataset.items);
                    
                    showPaymentModal(supplierId, supplierName, totalAmount, paidAmount, items);
                });
            });
        });

        function showPaymentModal(supplierId, supplierName, totalAmount, paidAmount, items) {
            try {
                // Ensure numbers
                totalAmount = parseFloat(totalAmount) || 0;
                paidAmount = parseFloat(paidAmount) || 0;
                const balance = totalAmount - paidAmount;

                // Filter unpaid items
                unpaidItems = items.filter(item => {
                    const total = parseFloat(item.quantity) * parseFloat(item.unit_price);
                    const isPaid = item.payment_status === 'paid';
                    const paidAmount = isPaid ? total : 0;
                    item.total = total;
                    item.paid = paidAmount;
                    item.remaining = total - paidAmount;
                    return item.remaining > 0;
                });

                // Reset the form first
                document.getElementById('paymentForm').reset();
                document.getElementById('mpesaSection').style.display = 'none';
                document.getElementById('chequeSection').style.display = 'none';

                // Set values with animation
                document.getElementById('payment_supplier_id').value = supplierId;
                document.getElementById('total_balance').textContent = balance.toFixed(2);
                document.getElementById('payment_amount').max = balance;
                document.getElementById('payment_amount').value = '';

                // Update payment distribution table
                updatePaymentDistributionTable();

                // Show modal with animation
                const modal = new bootstrap.Modal(document.getElementById('paymentModal'));
                modal.show();
                
                // Add entrance animation
                setTimeout(() => {
                    document.getElementById('paymentModal').classList.add('fade-in');
                }, 100);
                
            } catch (error) {
                console.error('Error in showPaymentModal:', error);
                showNotification('Error loading payment details. Please try again.', 'error');
            }
        }

        function handlePaymentMethodChange() {
            const method = document.getElementById('payment_method').value;
            const mpesaSection = document.getElementById('mpesaSection');
            const chequeSection = document.getElementById('chequeSection');
            
            mpesaSection.style.display = method === 'mpesa' ? 'block' : 'none';
            chequeSection.style.display = method === 'cheque' ? 'block' : 'none';
            
            // Make fields required based on payment method
            document.getElementById('mpesa_code').required = method === 'mpesa';
            document.getElementById('cheque_number').required = method === 'cheque';
        }

        function updatePaymentDistribution() {
            try {
                const paymentAmount = parseFloat(document.getElementById('payment_amount').value) || 0;
                let remainingPayment = paymentAmount;
                
                // Reset all payment amounts
                document.querySelectorAll('.payment-amount').forEach(cell => {
                    cell.textContent = '0.00';
                });
                
                // Distribute payment across items
                unpaidItems.forEach((item, index) => {
                    if (remainingPayment <= 0) return;
                    
                    const paymentAmount = Math.min(remainingPayment, item.remaining);
                    remainingPayment -= paymentAmount;
                    
                    const cells = document.querySelectorAll('.payment-amount');
                    if (cells[index]) {
                        cells[index].textContent = paymentAmount.toFixed(2);
                    }
                });
                
                // Update total payment amount
                const totalDistributed = paymentAmount - remainingPayment;
                document.getElementById('payment_amount').value = totalDistributed.toFixed(2);
            } catch (error) {
                console.error('Error in updatePaymentDistribution:', error);
            }
        }

        function updatePaymentDistributionTable() {
            const tbody = document.querySelector('#paymentDistributionTable tbody');
            tbody.innerHTML = '';
            
            if (!unpaidItems || unpaidItems.length === 0) {
                const row = document.createElement('tr');
                row.innerHTML = '<td colspan="6" class="text-center">No unpaid items found</td>';
                tbody.appendChild(row);
                return;
            }
            
            unpaidItems.forEach(item => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${item.created_at}</td>
                    <td>${item.product_name}</td>
                    <td>${item.total.toFixed(2)}</td>
                    <td>${item.paid.toFixed(2)}</td>
                    <td>${item.remaining.toFixed(2)}</td>
                    <td class="payment-amount">0.00</td>
                `;
                tbody.appendChild(row);
            });
        }

        function submitPayment() {
            try {
                const form = document.getElementById('paymentForm');
                const formData = new FormData(form);
                
                // Validate payment amount
                const amount = parseFloat(formData.get('amount'));
                const totalBalance = parseFloat(document.getElementById('total_balance').textContent);
                
                if (amount > totalBalance) {
                    document.getElementById('payment_amount').classList.add('is-invalid');
                    return;
                }
                
                // Validate required fields based on payment method
                const method = formData.get('payment_method');
                if (!method) {
                    document.getElementById('errorModalMessage').textContent = 'Please select a payment method';
                    var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                    errorModal.show();
                    return;
                }
                
                if (method === 'mpesa' && !formData.get('mpesa_code')) {
                    document.getElementById('errorModalMessage').textContent = 'Please enter M-PESA transaction code';
                    var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                    errorModal.show();
                    return;
                }
                if (method === 'cheque' && !formData.get('cheque_number')) {
                    document.getElementById('errorModalMessage').textContent = 'Please enter cheque number';
                    var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                    errorModal.show();
                    return;
                }
                
                // Add payment distribution to form data
                const paymentDistribution = [];
                document.querySelectorAll('#paymentDistributionTable tbody tr').forEach((row, index) => {
                    const amount = parseFloat(row.querySelector('.payment-amount').textContent);
                    if (amount > 0) {
                        paymentDistribution.push({
                            item_id: unpaidItems[index].id,
                            amount: amount
                        });
                    }
                });
                
                if (paymentDistribution.length === 0) {
                    alert('Please enter a payment amount');
                    return;
                }
                
                // Always send payment_status
                formData.append('payment_status', 'paid');
                formData.append('payment_distribution', JSON.stringify(paymentDistribution));
                
                // Show loading state
                const submitButton = document.querySelector('#paymentModal .modal-footer .btn-primary');
                const originalText = submitButton.innerHTML;
                submitButton.disabled = true;
                submitButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Processing...';
                
                // Submit payment
                fetch('process_payment.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Show Bootstrap success modal
                        document.getElementById('successModalMessage').textContent = 'Payment recorded successfully!';
                        var successModal = new bootstrap.Modal(document.getElementById('successModal'));
                        successModal.show();
                    } else {
                        document.getElementById('errorModalMessage').textContent = data.message || 'An error occurred while processing the payment.';
                        var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                        errorModal.show();
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    document.getElementById('errorModalMessage').textContent = 'Error: ' + error.message;
                    var errorModal = new bootstrap.Modal(document.getElementById('errorModal'));
                    errorModal.show();
                })
                .finally(() => {
                    // Reset button state
                    submitButton.disabled = false;
                    submitButton.innerHTML = originalText;
                });
            } catch (error) {
                console.error('Error in submitPayment:', error);
                alert('An error occurred while processing the payment. Please try again.');
            }
        }

        function editSupplier(supplier) {
            // Add animation to modal
            const modal = new bootstrap.Modal(document.getElementById('supplierModal'));
            modal.show();
            
            // Populate form with animation
            setTimeout(() => {
            document.getElementById('supplier_id').value = supplier.id;
            document.getElementById('name').value = supplier.name;
            document.getElementById('contact_person').value = supplier.contact_person;
            document.getElementById('phone').value = supplier.phone;
            document.getElementById('email').value = supplier.email;
            document.getElementById('address').value = supplier.address;
            document.getElementById('tax_number').value = supplier.tax_number;
                
                // Update modal title
            document.getElementById('supplierModalLabel').textContent = 'Edit Supplier';
                document.querySelector('#supplierModal button[type="submit"]').textContent = 'Update Supplier';
                
                // Add success animation
                document.getElementById('supplierModal').classList.add('fade-in');
            }, 100);
        }

        let supplierToDelete = null;

        function confirmDeleteSupplier(supplierId, supplierName, totalAmount, paidAmount) {
            const balance = totalAmount - paidAmount;
            const modal = new bootstrap.Modal(document.getElementById('deleteConfirmModal'));
            
            document.getElementById('deleteSupplierName').textContent = supplierName;
            
            if (balance > 0) {
                document.getElementById('deleteWarning').style.display = 'block';
                document.getElementById('deleteConfirmation').style.display = 'none';
                document.getElementById('amountOwed').textContent = `Amount owed: ${balance.toFixed(2)}`;
                document.getElementById('deleteButton').disabled = true;
            } else {
                document.getElementById('deleteWarning').style.display = 'none';
                document.getElementById('deleteConfirmation').style.display = 'block';
                document.getElementById('deleteButton').disabled = false;
            }
            
            modal.show();
        }

        function deleteSupplier() {
            if (!supplierToDelete) return;

            const formData = new FormData();
            formData.append('supplier_id', supplierToDelete);
            formData.append('delete_supplier', '1');

            fetch('suppliers.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(() => {
                window.location.reload();
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while deleting the supplier.');
            });
        }

        let currentPrintSupplierId = null;
        let currentPrintSupplierName = '';
        function openPrintModal(supplierId, supplierName) {
            currentPrintSupplierId = supplierId;
            currentPrintSupplierName = supplierName;
            document.getElementById('printSupplierName').textContent = supplierName;
            new bootstrap.Modal(document.getElementById('printOptionsModal')).show();
        }
        document.getElementById('printReportBtn').onclick = function() {
            if (currentPrintSupplierId) {
                window.open('/POS/pages/print_supplier_report.php?supplier_id=' + currentPrintSupplierId, '_blank');
            }
        };
        document.getElementById('printPaymentHistoryBtn').onclick = function() {
            if (currentPrintSupplierId) {
                window.open('supplier_payment_history.php?supplier_id=' + currentPrintSupplierId + '&print=1', '_blank');
            }
        };

        document.getElementById('addItemRow').addEventListener('click', function() {
            const table = document.getElementById('itemsTable').querySelector('tbody');
            const newRow = table.rows[0].cloneNode(true);
            newRow.querySelectorAll('input, select').forEach(input => input.value = '');
            table.appendChild(newRow);
        });
        document.getElementById('itemsTable').addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-item')) {
                const row = e.target.closest('tr');
                if (row.parentNode.rows.length > 1) row.remove();
            }
        });

        // Add this to your existing JavaScript
        document.addEventListener('DOMContentLoaded', function() {
            // Add animation to payment method cards
            const paymentMethod = document.getElementById('payment_method');
            paymentMethod.addEventListener('change', function() {
                const mpesaSection = document.getElementById('mpesaSection');
                const chequeSection = document.getElementById('chequeSection');
                
                // Reset sections
                mpesaSection.style.display = 'none';
                chequeSection.style.display = 'none';
                
                // Show selected section with animation
                if (this.value === 'mpesa') {
                    mpesaSection.style.display = 'block';
                    mpesaSection.style.opacity = '0';
                    setTimeout(() => {
                        mpesaSection.style.opacity = '1';
                    }, 50);
                } else if (this.value === 'cheque') {
                    chequeSection.style.display = 'block';
                    chequeSection.style.opacity = '0';
                    setTimeout(() => {
                        chequeSection.style.opacity = '1';
                    }, 50);
                }
            });
            
            // Remove the input formatting for amount to allow large numbers
            const paymentAmount = document.getElementById('payment_amount');
            paymentAmount.addEventListener('input', function() {
                // Only format if there's a decimal point
                if (this.value.includes('.')) {
                    const parts = this.value.split('.');
                    if (parts[1].length > 2) {
                        this.value = parseFloat(this.value).toFixed(2);
                    }
                }
            });
        });

        // Add hover effects to buttons
        document.addEventListener('DOMContentLoaded', function() {
            // Add ripple effect to buttons
            document.querySelectorAll('.btn').forEach(button => {
                button.addEventListener('click', function(e) {
                    const ripple = document.createElement('span');
                    const rect = this.getBoundingClientRect();
                    const size = Math.max(rect.width, rect.height);
                    const x = e.clientX - rect.left - size / 2;
                    const y = e.clientY - rect.top - size / 2;
                    
                    ripple.style.width = ripple.style.height = size + 'px';
                    ripple.style.left = x + 'px';
                    ripple.style.top = y + 'px';
                    ripple.classList.add('ripple');
                    
                    this.appendChild(ripple);
                    
                    setTimeout(() => {
                        ripple.remove();
                    }, 600);
                });
            });

            // Add loading states
            document.querySelectorAll('form').forEach(form => {
                form.addEventListener('submit', function() {
                    const submitBtn = this.querySelector('button[type="submit"]');
                    if (submitBtn) {
                        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
                        submitBtn.disabled = true;
                    }
                });
            });

            // Add tooltips
            document.querySelectorAll('[data-bs-toggle="tooltip"]').forEach(element => {
                new bootstrap.Tooltip(element);
            });

            // Add smooth scrolling
            document.querySelectorAll('a[href^="#"]').forEach(anchor => {
                anchor.addEventListener('click', function (e) {
                    e.preventDefault();
                    const target = document.querySelector(this.getAttribute('href'));
                    if (target) {
                        target.scrollIntoView({
                            behavior: 'smooth',
                            block: 'start'
                        });
                    }
                });
            });
        });

        // Enhanced notification system
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
            notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px; box-shadow: var(--shadow-lg);';
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-triangle' : 'info-circle'} me-2"></i>
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(notification);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.remove();
                }
            }, 5000);
        }

        // Enhanced form validation
        document.addEventListener('DOMContentLoaded', function() {
            // Add real-time validation
            document.querySelectorAll('.form-control').forEach(input => {
                input.addEventListener('blur', function() {
                    validateField(this);
                });
                
                input.addEventListener('input', function() {
                    if (this.classList.contains('is-invalid')) {
                        validateField(this);
                    }
                });
            });
        });

        function validateField(field) {
            const value = field.value.trim();
            const type = field.type;
            const required = field.hasAttribute('required');
            
            if (required && !value) {
                field.classList.add('is-invalid');
                return false;
            }
            
            if (type === 'email' && value && !isValidEmail(value)) {
                field.classList.add('is-invalid');
                return false;
            }
            
            if (type === 'tel' && value && !isValidPhone(value)) {
                field.classList.add('is-invalid');
                return false;
            }
            
            field.classList.remove('is-invalid');
            field.classList.add('is-valid');
            return true;
        }

        function isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        function isValidPhone(phone) {
            const phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
            return phoneRegex.test(phone.replace(/[\s\-\(\)]/g, ''));
        }
    </script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html> 